<?php
/*

Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK.

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

DFTK is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with DFTK; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright	Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK Trace Manager Class
 *
 * @package		dftk
 * @subpackage	dftk-trace
 * @author		Duck <duck@DuckCorp.org>
 *
 * @access		public
 */
class DftkDaTraceManager
{
	/**
	* List of directories where to find traces information
	*
	* @access		private
	* @var			array
	*/
	var		$_dirlist;

	/**
	 * Information about all registered events
	 *
	 * @access		private
	 * @var			array
	 */
	var		$_events;

	/**
	 * Tree of modules/sections for registered events
	 *
	 * @access		private
	 * @var			array
	 */
	var		$_events_tree;

	/**
	 * Language Manager
	 *
	 * @access		private
	 * @var			object DftkDaLanguageManager
	 */
	var		$_lm;

	/**
	 * Output object
	 *
	 * @access		private
	 * @var			object DftkOutput
	 */
	var $_output;

	/**
	 * Constructor
	 *
	 * @access		public
	 * @param		object DftkDaLanguageManager	&$lm			Language Manager
	 */
	function DftkDaTraceManager(&$lm)
	{
		$this->_dirlist = array();
		$this->_events = array();
		$this->_events_tree = array();
		$this->_lm =& $lm;
		$this->_output = null;
	}

	/**
	 * Set Output
	 *
	 * Set output to 'null' if no output wanted (default).
	 *
	 * @access		public
	 * @param		object DftkOutput			$output				Output object
	 */
	function set_output(&$output)
	{
		$this->_output =& $output;
	}

	/**
	 * Has Output ?
	 *
	 * @access		public
	 * @return		boolean						$r					true if Output defined
	 */
	function has_output()
	{
		if ($this->_output === null)
			return false;

		return true;
	}

	/**
	 * Gen Output message
	 *
	 * @access		private
	 * @param		string						$content			Message content
	 * @return		object DftkDaTrace			$r					Trace
	 */
	function &_gen_output($content)
	{
		$this->_output->clear_content();
		$this->_output->add_content($content);
		return $this->_output->flush_output();
	}

	/**
	 * Rebuild the list of events and the modules/sections tree
	 *
	 * @access		public
	 * @return		boolean					$r						true if reload was successful
	 */
	function reload()
	{
		$r = true;

		$this->_events = array();
		$this->_events_tree = array();
		foreach ($this->_dirlist as $dir)
		{
			if (!$this->_load($dir['module'], $dir['dir']))
				$r = false;
		}

		return $r;
	}

	/**
	 * Register traces
	 *
	 * @access		public
	 * @deprecated	Use 'register_events' instead, this was a confusing method name.
	 * @param		string					$module					Module name
	 * @param		string					$dir					Directory path where to find traces information
	 * @return		boolean					$r						true if registration was successful
	 */
	function register_traces($module, $dir)
	{
		return $this->register_events($module, $dir);
	}

	/**
	 * Register events
	 *
	 * @access		public
	 * @param		string					$module					Module name
	 * @param		string					$dir					Directory path where to find traces information
	 * @return		boolean					$r						true if registration was successful
	 */
	function register_events($module, $dir)
	{
		if (file_exists($dir))
		{
			$this->_dirlist[] = array('module' => $module, 'dir' => $dir);
			return $this->_load($module, $dir);
		}
		else
			return false;
	}

	/**
	 * Load traces information
	 *
	 * @access		private
	 * @param		string					$module					Module name
	 * @param		string					$dir					Directory path where to find traces information
	 * @return		boolean					$r						true if load was successful
	 */
	function _load($module, $dir)
	{
		if (file_exists($dir."/msgs") && file_exists($dir."/events"))
		{
			$this->_lm->add_trans($dir."/msgs");

			$handle = opendir($dir."/events");
			while ($file = readdir($handle))
			{
				if ($file=='.' || $file=='..')
					continue;
				if (is_dir($file))
					continue;
				$z = parse_ini_file($dir."/events/".$file, true);
				$this->_load_events($module, null, $z);
			}
			closedir($handle);

			return true;
		}
		else
			return false;
	}

	/**
	 * Load events information
	 *
	 * @access		private
	 * @param		string					$modules				Modules name
	 * @param		string					$section				Section name
	 * @param		array					$tab					Events information
	 */
	function _load_events($module, $section, $tab)
	{
				foreach ($tab as $key => $trace)
				{
					if (is_array($trace))
						$this->_load_events($module, $key, $trace);
					else
					{
						$params = preg_split("/\|/", $trace, 1);
						if (!preg_match("/^[EWN]$/", $params[0]))
							continue;
						
						$this->_events[$key]['module'] = $module;
						$this->_events[$key]['section'] = $section;
						$this->_events[$key]['severity'] = $params[0];

						if ($section == null)
						{
							if (!$this->_events_tree[$module])
								$this->_events_tree[$module] = array();
						}
						else
							$this->_events_tree[$module][] = $section;
					}
				}
	}

	/**
	 * Get event information
	 *
	 * Event information fields :
	 *  - modules : module name in which the event is registered
	 *  - section : section name in which the event is registered
	 *  - severity : event severity
	 *
	 * List of severities :
	 *  - E : error
	 *  - W : warning
	 *  - N : notice
	 *
	 * @access		public
	 * @param		string					$key					Event Id
	 * @return		array					$r						Information about event (or false if event not existing)
	 */
	function get_event_infos($key)
	{
		if ($this->_events[$key])
			return $this->_events[$key];

		return false;
	}

	/**
	 * Get the event readable message
	 *
	 * @access		private
	 * @param		string					$key					Event Id
	 * @param		array					$params					Parameters of the message
	 * @return		string					$r						Message (or false if event not existing)
	 */
	function _get_event_msg($key, $params)
	{
		if ($this->_events[$key])
			return $this->_lm->get_msg($key, $params);

		return false;
	}


	/**
	 * Get the event message template
	 *
	 * @access		public
	 * @param		string					$key					Event Id
	 * @return		string					$r						Message template (or false if event not existing)
	 */
	function get_event_msg_template($key)
	{
		if ($this->_events[$key])
			return $this->_lm->get_msg($key, null);

		return false;
	}

	/**
	 * Test if a key is an event Id
	 *
	 * @access		public
	 * @param		string					$key					Event Id
	 * @return		boolean					$r						true if the key is an event Id
	 */
	function is_event($key)
	{
		if ($this->_events[$key])
			return true;

		return false;
	}

	/**
	 * Get the selected Language Manage
	 *
	 * @access		public
	 * @return		object DftkDaLanguageManager	$r						Language Manage
	 */
	function &get_language_manager()
	{
		return $this->_lm;
	}

	/**
	 * Get a list of registered events
	 *
	 * @access		public
	 * @return		array					$tab					List of registered events
	 */
	function get_event_list()
	{
		$tab = array();
		foreach ($this->_events as $key => $val)
			$tab[] = $key;

		return $tab;
	}

	/**
	 * Get a list of existing modules
	 *
	 * @access		public
	 * @return		array					$tab					List of existing modules
	 */
	function get_module_list()
	{
		$tab = array();
		foreach ($this->_events_tree as $key => $val)
			$tab[] = $key;

		return $tab;
	}

	/**
	 * Test if a string is a module name
	 *
	 * @access		public
	 * @param		string					$module					Module name
	 * @return		boolean					$r						true if the string is a module name
	 */
	function is_module($module)
	{
		foreach ($this->_events_tree as $key => $val)
			if ($key == $module)
				return true;

		return false;
	}

	/**
	 * Get a list of existing sections
	 *
	 * @access		public
	 * @return		array					$tab					List of existing sections
	 */
	function get_section_list($module)
	{
		if (!$this->is_module($module))
			return false;

		foreach ($this->_events_tree[$module] as $key)
			$tab[] = $key;

		return $tab;
	}

	/**
	 * Test if a string is a section name
	 *
	 * @access		public
	 * @param		string					$section				Section name
	 * @return		boolean					$r						true if the string is a section name
	 */
	function is_section($module, $section)
	{
		if (!$this->is_module($module))
			return false;

		foreach ($this->_events_tree[$module] as $key)
			if ($key == $section)
				return true;

		return false;
	}

	/**
	 * Creste a new Trace attached to this Manager
	 *
	 * @access		public
	 * @return		object DftkDaTrace		$z						Created Trace
	 */
	function &create_trace()
	{
		$z =& new DftkDaTrace(&$this);
		return $z;
	}
}

?>
