<?php
/*

Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK.

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

DFTK is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with DFTK; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright	Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK Trace Class
 *
 * @package		dftk
 * @subpackage	dftk-trace
 * @author		Duck <duck@DuckCorp.org>
 *
 * @access		public
 */
class DftkDaTrace
{
	/**
	 * Trace Manager
	 *
	 * @access		private
	 * @var			object DftkDaTraceManager
	 */
	var		$_manager;

	/**
	 * List of Events
	 *
	 * @access		private
	 * @var			array
	 */
	var		$_events;

	/**
	 * List of Results
	 *
	 * @access		private
	 * @var			array
	 */
	var		$_results;

	/**
	 * Constructor
	 *
	 * @access		public
	 * @param		object DftkDaTraceManager		&$manager				Trace Manager
	 */
	function DftkDaTrace(&$manager)
	{
		$this->_manager =& $manager;
		$this->_events = array();
		$this->_results = array();
	}

	/**
	 * Add an Event to the list
	 *
	 * @access		public
	 * @param		string				$key				Event Id
	 * @param		array				$parals				List of message parameters
	 */
	function add_event($key, $params = array())
	{
		if (!$this->_manager->is_event($key))
			return false;

		if (!is_array($params))
			$params = array($params);
		$tab = array();
		$tab['timestamp'] = date("Y-m-d\TH:i:s");
		$tab['event'] = $key;
		$tab['params'] = $params;
		$this->_events[] = $tab;
	}

	/**
	 * Remove every result in the list
	 *
	 * @access		public
	 */
	function clear_results()
	{
		$this->_results = array();
	}

	/**
	 * Unset a result's value
	 *
	 * @access		public
	 * @param		string				$key				Result Id
	 */
	function unset_result($key)
	{
		unset($this->_results[$key]);
	}

	/**
	 * Set a result's value
	 *
	 * @access		public
	 * @param		string				$key				Result Id
	 * @param		mixed				$obj				Value
	 */
	function set_result($key, $obj)
	{
		$this->_results[$key] =& $obj;
	}

	/**
	 * Get a result's value
	 *
	 * @access		public
	 * @param		string				$key				Result Id
	 * @return		mixed				$r					Value
	 */
	function &get_result($key)
	{
		if ($this->_results[$key])
			return $this->_results[$key];

		return false;
	}

	/**
	 * Search for events
	 *
	 * List of severities :
	 *  - E : error
	 *  - W : warning
	 *  - N : notice
	 *
	 * Each element of the returned array contains these entries :
	 *  - event : event Id
	 *  - timestamp : ISO8601 date+time format when the event occured
	 *  - module : module name in which the event occured
	 *  - section : section name in which the event occured
	 *  - msg : readable message
	 *  - fullmsg : readable message with full information
	 *
	 * @access		public
	 * @param		string				$severity			Severity criteria ( 'E'/'W'/'N' or null if not a criteria)
	 * @param		string				$module				Module name criteria (or null if not a criteria)
	 * @param		string				$section			Section name criteria (or null if not a criteria) (if a criteria, $module must not be null)
	 * @return		array				$ret				Array of found events
	 */
	function get_events($severity = null, $module = null, $section = null)
	{
		if ($severity != null)
			if (!preg_match("/^[EWN]$/", $severity))
				return array();

		if ($module != null)
		{
			if ($section != null)
			{
				if (!$this->_manager->is_section($module, $section))
					return array();
			}
			else
			{
				if (!$this->_manager->is_module($module))
					return array();
			}
		}

		$ret = array();

		foreach ($this->_events as $ev)
		{
			$info = $this->_manager->get_event_infos($ev['event']);

			if ($severity != null)
				if ($info['severity'] != $severity)
					continue;
			if ($module != null)
			{
				if ($info['module'] != $module)
					continue;

				if ($section != null)
					if ($info['section'] != $section)
						continue;
			}

			$tab = array();
			$tab['event'] = $ev['event'];
			$tab['timestamp'] = $ev['timestamp'];
			$tab['module'] = $info['module'];
			$tab['section'] = $info['section'];
			$tab['msg'] = $this->_manager->_get_event_msg($ev['event'], $ev['params']);
			if ($tab['section'])
				$tab['fullmsg'] = $tab['timestamp']."   ".$tab['module']."::".$tab['section']."::".$tab['event']."   '".$tab['msg']."'";
			else
				$tab['fullmsg'] = $tab['timestamp']."   ".$tab['module']."::".$tab['event']."   '".$tab['msg']."'";
			$ret[] = $tab;
		}

		return $ret;
	}

	/**
	 * Create a debug message
	 *
	 * @access		public
	 * @param		boolean				$html				true for HTML output, false for plain text
	 * @return		string				$msg
	 */
	function get_debug_msg($html = false)
	{
		$msg = "";
		if ($html)
			$msg = "<table border='1'><tr><th>Timestamp</th><th>Severity</th><th>Module</th><th>Section</th><th>Message</th></tr>";

		foreach ($this->_events as $ev)
		{
			$info = $this->_manager->get_event_infos($ev['event']);

			$tab = array();
			$tab['event'] = $ev['event'];
			$tab['timestamp'] = $ev['timestamp'];
			$tab['module'] = $info['module'];
			$tab['section'] = $info['section'];
			$tab['severity'] = $info['severity'];
			$tab['msg'] = $this->_manager->_get_event_msg($ev['event'], $ev['params']);

			if ($html)
				$msg.="<tr><td>".$tab['timestamp']."</td><td align='center'>".$tab['severity']."</td><td align='center'>".$tab['module']."</td><td align='center'>".$tab['section']."</td><td>".$tab['msg']."</td></tr>\n";
			else
			{
				if ($tab['section'])
					$tab['fullmsg'] = $tab['timestamp']."   ".$tab['severity']."   ".$tab['module']."::".$tab['section']."   '".$tab['msg']."'";
				else
					$tab['fullmsg'] = $tab['timestamp']."   ".$tab['severity']."   ".$tab['module']."   '".$tab['msg']."'";

				if ($msg == "")
					$msg=$tab['fullmsg'];
				else
					$msg.="\n".$tab['fullmsg'];
			}
		}

		if ($html)
			$msg.="</table>";

		return $msg;
	}

	/**
	 * Generate output
	 *
	 * @access		public
	 * @return		object DftkDaTrace			$r					Trace
	 */
	function &gen_output()
	{
		if (!$this->_manager->has_output())
			return;

		$content = "";
		foreach ($this->_events as $ev)
			$content.= $this->_manager->_get_event_msg($ev['event'], $ev['params'])."\n";

		return $this->_manager->_gen_output($content);
	}

	/**
	 * Look if a special event has occured
	 *
	 * @access		public
	 * @param		string				$key					Event Id
	 * @return		boolean				$r						true if event occured
	 */
	function test_if_event($key)
	{
		if (!$this->_manager->is_event($key))
			return false;

		foreach ($this->_events as $ev)
			if ($ev['event'] == $key)
				return true;

		return false;
	}

	/**
	 * Merge Traces
	 *
	 * Merge current trace with given trace :
	 *  - results are merged (beware of collisions)
	 *  - events are sort-merged
	 *
	 * @access		public
	 * @param		object DftkDaTrace	&$trace
	 */
	function merge_traces(&$trace)
	{
		// Buggy check, deactivated
		// (in some cases using a different manager might be useful
		//  without being inconsistent)
		//if ($this->_manager == $trace->_manager)
		//{
			$this->_results = array_merge($this->_results, $trace->_results);

			$tab = array_merge($this->_events, $trace->_events);
			$this->_events = array();
			if (count($tab))
			{
				$tab2 = array();
				for ($i=0; $i<count($tab); $i++)
				{
					$tab2[0][] = $tab[$i]['timestamp'];
					$tab2[1][] = $i;
				}
				array_multisort($tab2[0], $tab2[1]);
				for ($i=0; $i<count($tab2[1]); $i++)
					$this->_events[] = $tab[$tab2[1][$i]];
			}
		//}
	}

	/**
	 * Test if events with an 'error' severity has occured
	 *
	 * @access		public
	 * @return		boolean				$r						true if such event occured
	 */
	function has_error()
	{
		if (count($this->get_events('E'))>0)
			return true;

		return false;
	}

	/**
	 * Test if events with an 'warning' severity has occured
	 *
	 * @access		public
	 * @return		boolean				$r						true if such event occured
	 */
	function has_warning()
	{
		if (count($this->get_events('W'))>0)
			return true;

		return false;
	}
}

?>
