<?php
/*

Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK.

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

DFTK is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with DFTK; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright	Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK LDAP Entries Class
 *
 * @package		dftk
 * @subpackage	dftk-ldap
 * @author		Duck <duck@DuckCorp.org>
 *
 * @access		public
 */
class DftkLdapEntries extends DftkLdapBase
{
	/**
	 * LDAP Result
	 *
	 * @access		private
	 * @var			array
	 */
	var	$_tab;
	/**
	 * Number of results
	 *
	 * @access		private
	 * @var			integer
	 */
	var	$_nb;
	/**
	 * DN for each result
	 *
	 * @access		private
	 * @var			array
	 */
	var	$_dn;

	/**
	 * Constructor
	 *
	 * @access		public
	 * @param		object DftkDaLanguageManager	$tracemgr			Language Manager
	 */
	function DftkLdapEntries(&$tracemgr)
	{
		DftkLdapBase::DftkLdapBase($tracemgr);

		$this->clear();

		register_shutdown_function(array(&$this, "_DftkLdapEntries"));
	}

	/**
	 * Destructor
	 *
	 * @access	private
	 */
	function _DftkLdapEntries()
	{
	}

	/**
	 * Destroy results
	 *
	 * @access		public
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &clear()
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_tab = array();
		$this->_nb = 0;
		$this->_dn = array();

		return $r;
	}

	/**
	 * Put LDAP result into object
	 *
	 * @access		private
	 * @param		array				$tab				Array containing LDAP results
	 * @param		integer				$nb					Number of LDAP results
	 */
	function _put_tab($tab, $nb)
	{
		$this->clear();
		$this->_tab = $tab;
		$this->_nb = $nb;
		$this->_process_entries_read();
	}

	/**
	 * Export LDAP entries for writing into Database
	 *
	 * @access		private
	 * @param		integer				$i					Number of the entry wanted
	 * @param		array				$attr_list			List of Attributs to export (all if empty)
	 * @return		array				$tab				Array containing LDAP entry ready to be written
	 */
	function _export_entry($i, $attr_list = array())
	{
		$tab = $this->_tab[$i];
		
		foreach ($tab as $key1 => $elem1)
		{
			if (is_integer($key1) || (count($attr_list)!=0 && !in_array($key1, $attr_list)))
			{
				unset($tab[$key1]);
			}
			else if (is_array($elem1))
				foreach ($elem1 as $key2 => $elem2)
				{
					if (is_string($elem2))
						$tab[$key1][$key2] = utf8_encode($elem2);
				}
		}
		
		return $tab;
	}

	/**
	 * Give Nuber of entries
	 *
	 * @access		public
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &get_count()
	{
		$r =& $this->_tracemgr->create_trace();

		$r->set_result('count', $this->_nb);
		
		return $r;
	}

	/**
	 * Give a reference on an array containing one particular entry
	 *
	 * @access		public
	 * @param		integer				$i					Number of the entry wanted
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &get_entry($i)
	{
		$r =& $this->_tracemgr->create_trace();

		//return $this->_tab[$i];
		$r->set_result('entry', &$this->_tab[$i]);

		return $r;
	}

	/**
	 * Give the DN of a particular result
	 *
	 * @access		public
	 * @param		integer				$i					Number of the entry wanted
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &get_dn($i)
	{
		$r =& $this->_tracemgr->create_trace();

		$r->set_result('dn', $this->_dn[$i]);

		return $r;
	}

	/**
	 * Set the DN of a particular result (for rename remembering purpose)
	 *
	 * @access		public
	 * @param		integer				$i					Number of the entry wanted
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &set_dn($i, $new_dn)
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_dn[$i] = $new_dn;

		return $r;
	}

	/**
	 * Change one entry
	 *
	 * @access		public
	 * @param		integer				$i					Number of the entry to change
	 * @param		array				$entry				Array containing the data of the entry to change
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &set_entry($i, $entry)
	{
		$r =& $this->_tracemgr->create_trace();

		if (is_array($entry))
			$this->_tab[$i] = $entry;
		else
			$r->add_event('dftk-ldap_setentbade', $i);

		return $r;
	}

	/**
	 * Sort entries using one or two attributs
	 *
	 * @access		public
	 * @param		string				$attr1				Key for the first attribut
	 * @param		string				$attr2				Key for the second attribut or null if unused
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &sort_entries_by_attr($attr1, $attr2 = null)
	{
		$r =& $this->_tracemgr->create_trace();

		$temp_sort1 = array();
		if ($attr2!=null)
			$temp_sort2 = array();
		for ($i=0; $i<count($this->_tab); $i++)
		{
			$temp_sort1[$i] = $this->_tab[$i][$attr1][0];
			if ($attr2!=null)
				$temp_sort2[$i] = $this->_tab[$i][$attr2][0];
		}
		if ($attr2!=null)
			array_multisort($temp_sort1, $temp_sort2, $this->_tab);
		else
			array_multisort($temp_sort1, $this->_tab);

		return $r;
	}

	/**
	 * Process entries for use from LDAP
	 *
	 * @access		private
	 */
	function _process_entries_read()
	{
		unset($this->_tab['count']);
		foreach ($this->_tab as $key1 => $elem1)
		{
			unset($this->_tab[$key1]['count']);
			$this->_dn[$key1] = $this->_tab[$key1]['dn'];
			unset($this->_tab[$key1]['dn']);
			foreach ($this->_tab[$key1] as $key2 => $elem2)
			{
				unset($this->_tab[$key1][$key2]['count']);
				if (is_array($this->_tab[$key1][$key2]))
					foreach ($this->_tab[$key1][$key2] as $key3 => $elem3)
					{
						if (is_string($elem3))
							$this->_tab[$key1][$key2][$key3] = utf8_decode($elem3);
					}
			}
		}
	}

	/**
	 * Display Entries (for debugging purpose)
	 *
	 * @access		public
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &display_entries()
	{
		$r =& $this->_tracemgr->create_trace();

		$this->display_array($this->_tab);

		return $r;
	}

}

?>
