<?php
/*

Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK.

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

DFTK is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with DFTK; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright	Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK LDAP Connection Handling Class
 *
 * @package		dftk
 * @subpackage	dftk-ldap
 * @author		Duck <duck@DuckCorp.org>
 *
 * @access		public
 */
class DftkLdapConnection extends DftkLdapBase
{
	/**
	 * Connected to an LDAP server ?
	 *
	 * @access		private
	 * @var			boolean
	 */
	var	$_connected;
	/**
	 * LDAP Server
	 * @access		private
	 * @var			string
	 */
	var	$_server;
	/**
	 * LDAP Ressource
	 *
	 * @access		private
	 * @var			integer
	 */
	var	$_ds;
	/**
	 * Logged in ?
	 *
	 * @access		private
	 * @var			boolean
	 */
	var	$_binded;
	/**
	 * User Bind DN
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_bind_dn;
	/**
	 * User Bind Password
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_bind_passwd;
	/**
	 * User Type
	 *
	 * 0 => Unknown User Type
	 * 1 => Common User
	 * 2 => Admin User
	 * 3 => Service User
	 *
	 * @access		private
	 * @var			integer
	 */
	var	$_user_type;
	/**
	 * Helping parameters set ?
	 *
	 * @access		private
	 * @var			boolean
	 */
	var	$_has_wrapping_parameters;
	/**
	 * LDAP Base DN
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_base_dn;
	/**
	 * User part DN
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_user_part_dn;
	/**
	 * Common User part DN
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_commonuser_part_dn;
	/**
	 * Admin part User
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_adminuser_part_dn;
	/**
	 * Service User part DN
	 *
	 * @access		private
	 * @var			string
	 */
	var	$_serviceuser_part_dn;

	/**
	 * Constructor
	 *
	 * @access		public
	 * @param		object DftkDaTraceManager	&$tracemgr			Trace Manager
	 */
	function DftkLdapConnection(&$tracemgr)
	{
		DftkLdapBase::DftkLdapBase($tracemgr);
		
		$this->_init();
		
		register_shutdown_function(array(&$this, "_DftkLdapConnection"));
	}

	/**
	 * Destructor
	 *
	 * @access		private
	 */
	function _DftkLdapConnection()
	{
		if ($this->_connected)
			$this->disconnect();
	}

	/**
	 * Initialisation
	 *
	 * @access		private
	 */
	function _init()
	{
		$this->_server = "";
		$this->_connected = false;
		$this->_ds = 0;
		$this->_binded = false;
		$this->_bind_dn = "";
		$this->_bind_passwd = "";
		$this->_user_type = 0;
		$this->_has_wrapping_parameters = "";
		$this->_base_dn = "";
		$this->_user_part_dn = "";
		$this->_commonuser_part_dn = "";
		$this->_adminuser_part_dn = "";
		$this->_serviceuser_part_dn = "";
	}

	/**
	 * Connect to an LDAP server
	 *
	 * @access		public
	 * @param		string				$server				Server Name
	 * @param		boolean				$checkv3			Protocol version 3 needed ?
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &connect($server, $checkv3 = true)
	{
		$r =& $this->_tracemgr->create_trace();
		$this->_server = $server;
		
		if ($this->_connected)
		{
			$r2 =& $this->disconnect();
			$r->merge_traces(&$r2);
		}

		$this->_ds = @ldap_connect($this->_server);
		if ($this->_ds)
		{
			if ($checkv3 && !ldap_set_option($this->_ds, LDAP_OPT_PROTOCOL_VERSION, 3))
			{
				$r2 =& $this->disconnect();
				$r->merge_traces(&$r2);
				$r->add_event('dftk-ldap_badprotover');
			}
			else
				$this->_connected = true;
		}
		else
			$r->add_event('dftk-ldap_cannotconnect');

		return $r;
	}

	/**
	 * Disconnect from an LDAP server
	 *
	 * @access		public
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &disconnect()
	{
		$r =& $this->_prerequisite(false, false);
		
		if (!$r->has_error())
		{
			if (@ldap_close($this->_ds))
				$this->_init();
			else
				$r->add_event('dftk-ldap_probdisconnect');
		}
		else
			$r->add_event('dftk-ldap_cannotdisconnect');

		return $r;
	}

	/**
	 * Log into an ldap server
	 *
	 * @access		public
	 * @param		string				$bind_dn			dn of the user
	 * @param		string				$bind_passwd		password for the user
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &bind($bind_dn, $bind_passwd)
	{
		$r =& $this->_prerequisite(false, false);

		if (!$r->has_error())
		{
			if (@ldap_bind($this->_ds, $bind_dn, $bind_passwd))
			{
				$this->_binded =true;
				$this->_bind_dn = $bind_dn;
				$this->_bind_passwd = $bind_passwd;
				$this->_user_type = 0;
			}
			else
				$r->add_event('dftk-ldap_probbind', ldap_error($this->_ds));
		}

		return $r;
	}

	/**
	 * Set helping parameters
	 *
	 * @access		public
	 * @param		string				$base_dn				DN of the root node of LDAP
	 * @param		string				$user_part_dn			Part of the DN to add to find all the users
	 * @param		string				$commonuser_part_dn		Part of the DN to add to find the common users
	 * @param		string				$adminuser_part_dn		Part of the DN to add to find the admin users
	 * @param		string				$serviceuser_part_dn	Part of the DN to add to find the service users
	 * @return		object DftkDaTrace	$r						Trace
	 */
	function &set_wrapping_parameter($base_dn, $user_part_dn, $commonuser_part_dn, $adminuser_part_dn, $serviceuser_part_dn)
	{
		$r =& $this->_tracemgr->create_trace();
		
		$this->_has_wrapping_parameters = true;
		$this->_base_dn = $base_dn;
		$this->_user_part_dn = $user_part_dn;
		$this->_commonuser_part_dn = $commonuser_part_dn;
		$this->_adminuser_part_dn = $adminuser_part_dn;
		$this->_serviceuser_part_dn = $serviceuser_part_dn;

		return $r;
	}
	
	/**
	 * Log into an ldap server
	 *
	 * @access		public
	 * @param		string				$bind_user			UID for the user
	 * @param		string				$bind_passwd		password for the user
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &w_bind($bind_user, $bind_passwd)
	{
		$r =& $this->_prerequisite(false, true);
		if (!$r->has_error())
		{
			$this->_user_type = 0;

			$bind_dn = "uid=".$bind_user.",".$this->_commonuser_part_dn.",".$this->_user_part_dn.",".$this->_base_dn;
			if (!@ldap_bind($this->_ds, $bind_dn, $bind_passwd))
			{
				$bind_dn = "uid=".$bind_user.",".$this->_adminuser_part_dn.",".$this->_user_part_dn.",".$this->_base_dn;
				if (!@ldap_bind($this->_ds, $bind_dn, $bind_passwd))
				{
					$bind_dn = "uid=".$bind_user.",".$this->_serviceuser_part_dn.",".$this->_user_part_dn.",".$this->_base_dn;
					if (!@ldap_bind($this->_ds, $bind_dn, $bind_passwd))
					{
						$r->add_event('dftk-ldap_probbind');
						return false;
					}
					else
						$this->_user_type = 3;
				}
				else
					$this->_user_type = 2;
			}
			else
				$this->_user_type = 1;
		
			if ($this->_user_type)
			{
				$this->_binded =true;
				$this->_bind_dn = $bind_dn;
				$this->_bind_passwd = $bind_passwd;
			}
		}
			
		return $r;
	}

	/**
	 * Give the user type (Admin/Common/Service)
	 *
	 * @access		public
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &get_user_type()
	{
		$r =& $this->_prerequisite(true, false);

		if (!$r->has_error())
			$r->set_result('user_type', $this->_user_type);

		return $r;
	}

	/**
	 * Search in the LDAP Database
	 *
	 * @access		public
	 * @param		string					$base			DN where to begin the search
	 * @param		string					$search			Search pattern
	 * @param		object DftkLdapEntries	&$entries		Object where to put results
	 * @param		array					$attr			Array of attribut keys to fetch
	 * @param		boolean					$subtree		Search in subtrees ?
	 * @return		object DftkDaTrace		$r				Trace
	 */
	function &search($base, $search, &$entries, $attr = array(), $subtree = true)
	{
		$r =& $this->_prerequisite(true, false);
		if (!$r->has_error)
		{
			if ($subtree)
				$f = "ldap_search";
			else
				$f = "ldap_list";
			
			if (is_array($attr) && count($attr)>0)
				$sr = @$f($this->_ds, $base, $search, $attr);
			else
				$sr = @$f($this->_ds, $base, $search);

			if ($sr)
			{
				$nb = ldap_count_entries($this->_ds, $sr);
				if (!$nb)
					$tab = array();
				else
					$tab = ldap_get_entries($this->_ds, $sr);

				ldap_free_result($sr);
				$entries->_put_tab($tab, $nb);
				$r->set_result('count', $nb);
			}
			else
				$r->add_event('dftk-ldap_probsearch', ldap_error($this->_ds));
		}
		
		return $r;
	}

	/**
	 * Search in all the LDAP Database
	 *
	 * @access		public
	 * @param		string					$search			Search pattern
	 * @param		object DftkLdapEntries	&$entries		Object where to put results
	 * @param		array					$attr			Array of attribut keys to fetch
	 * @param		boolean					$subtree		Search in subtrees ?
	 * @return		object DftkDaTrace		$r				Trace
	 */
	function &w_search($search, &$entries, $attr = array(), $subtree = true)
	{
		$r =& $this->_prerequisite(true, true);
		if (!$r->has_error())
		{
			$r2 =& $this->search($this->_base_dn, $search, $entries, $attr);
			$r->merge_traces(&$r2);
		}

		return $r;
	}

	/**
	 * Modify an LDAP Object
	 *
	 * @access		public
	 * @param		string					$object			DN of the Object
	 * @param		object DftkLdapEntries	&$entries		Object data
	 * @param		integer					$i				Number of the entry to Replace
	 * @param		array					$attr_list		List of Attributes to Replace
	 * @return		object DftkDaTrace		$r				Trace
	 */
	function &modify($object, &$entries, $i, $attr_list = array())
	{
		$r =& $this->_prerequisite(true, false);
		if (!$r->has_error())
		{
			if (count($attr_list)!=0)
			{
				if (!@ldap_mod_replace($this->_ds, $object, $entries->_export_entry($i, $attr_list)))
					$r->add_event('dftk-ldap_probmod', ldap_error($this->_ds));
			}
			else
			{
				if (!@ldap_modify($this->_ds, $object, $entries->_export_entry($i)))
					$r->add_event('dftk-ldap_probmod', ldap_error($this->_ds));
			}
		}

		return $r;
	}

	/**
	 * Add an LDAP Object
	 *
	 * @access		public
	 * @param		string					$object			DN of the Object
	 * @param		object DftkLdapEntries	&$entries		Object data
	 * @param		integer					$i				Number of the entry to Replace
	 * @return		object DftkDaTrace		$r				Trace
	 */
	function &add($object, &$entries, $i)
	{
		$r =& $this->_prerequisite(true, false);
		if (!$r->has_error())
			if (!@ldap_add($this->_ds, $object, $entries->_export_entry($i)))
				$r->add_event('dftk-ldap_probadd', ldap_error($this->_ds));

		return $r;
	}

	/**
	 * Delete an LDAP Object
	 *
	 * @access		public
	 * @param		string				$object				DN of the Object
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &delete($object)
	{
		$r =& $this->_prerequisite(true, false);
		if (!$r->has_error())
		{
			//if (!$object)
			//	return false;

			if (!@ldap_delete($this->_ds, $object))
				$r->add_event('dftk-ldap_probdel', ldap_error($this->_ds));
		}

		return $r;
	}

	/**
	 * Rename an LDAP Object
	 *
	 * @access		public
	 * @param		string				$object				DN of the Object
	 * @param		string				$new_object			New DN of the Object
	 * @return		object DftkDaTrace	$r					Trace
	 */
	function &rename($object, $new_object)
	{
		$r =& $this->_prerequisite(true, false);
		if (!$r->has_error())
		{
			$z = ldap_explode_dn($new_object, 0);
			$newrdn = $z[0];
			array_splice($z, 0, 2);
			$newparent = implode(",", $z);
			if (!@ldap_rename($this->_ds, $object, $newrdn, $newparent, false))
				$r->add_event('dftk-ldap_probren', ldap_error($this->_ds));
		}

		return $r;
	}

	/**
	 * Check prerequisite for other methodes
	 *
	 * @access		private
	 * @param		boolean				$need_bind				Is login needed ?
	 * @param		boolean				$need_wrapping_params	Are helping parameters needed ?
	 * @return		object DftkDaTrace	$r						Trace
	 */
	function &_prerequisite($need_bind, $need_wrapping_params)
	{
		$r =& $this->_tracemgr->create_trace();

		if (!$this->_connected)
			$r->add_event('dftk-ldap_notconnected');
		else if ($need_bind && !$this->_binded)
			$r->add_event('dftk-ldap_notbinded');
		else if ($need_wrapping_params && !$this->_has_wrapping_parameters)
			$r->add_event('dftk-ldap_missparams');

		return $r;
	}
}

?>
